import Image from 'next/image'
import Link from 'next/link'

interface BlogPostPageProps {
  params: Promise<{
    slug: string
  }>
}

export async function generateStaticParams() {
  // Return all possible slug values for static generation
  return [
    { slug: 'complete-guide-auto-insurance-2025' },
    { slug: 'credit-score-insurance-rates' },
    { slug: 'what-to-do-after-car-accident' },
    { slug: 'understanding-insurance-deductibles' },
    { slug: 'lower-auto-insurance-premiums' },
    { slug: 'teen-driver-insurance-guide' },
    { slug: 'gap-insurance-explained' },
  ]
}

export default async function BlogPostPage({ params }: BlogPostPageProps) {
  const { slug } = await params
  // This would typically fetch the blog post data based on the slug
  // For now, we'll use static content
  const post = {
    title: 'The Complete Guide to Auto Insurance in 2025',
    excerpt: 'Everything you need to know about auto insurance coverage, costs, and how to find the best policy for your needs.',
    image: 'https://images.unsplash.com/photo-1549317336-206569e8475c?w=1200&h=600&fit=crop',
    author: 'Sarah Johnson',
    date: 'January 15, 2025',
    readTime: '8 min read',
    category: 'Insurance Basics',
    content: `
      <p>Auto insurance is one of those necessities in life that we all need but many of us don't fully understand. Whether you're a first-time car buyer or looking to switch providers, this comprehensive guide will help you navigate the complex world of auto insurance in 2025.</p>

      <h2>Understanding the Basics</h2>
      <p>Auto insurance is a contract between you and an insurance company that protects you against financial loss if you're in an accident or your car is stolen or damaged. In exchange for paying a premium, the insurance company agrees to pay your losses as outlined in your policy.</p>

      <h3>Types of Coverage</h3>
      <p>There are several types of auto insurance coverage, and understanding each is crucial to making informed decisions:</p>

      <ul>
        <li><strong>Liability Coverage:</strong> Required in most states, this covers damages you cause to others</li>
        <li><strong>Collision Coverage:</strong> Pays for damage to your car from crashes</li>
        <li><strong>Comprehensive Coverage:</strong> Covers damage from theft, vandalism, weather, and other non-collision events</li>
        <li><strong>Uninsured/Underinsured Motorist:</strong> Protects you if the other driver has insufficient insurance</li>
        <li><strong>Personal Injury Protection (PIP):</strong> Covers medical expenses regardless of who's at fault</li>
      </ul>

      <h2>How Much Coverage Do You Need?</h2>
      <p>The amount of coverage you need depends on several factors, including your state's requirements, your financial situation, and your risk tolerance. Here are some guidelines:</p>

      <h3>State Minimums vs. Recommended Coverage</h3>
      <p>While every state (except New Hampshire) requires some form of auto insurance, the minimum requirements are often insufficient to fully protect you. We recommend considering higher limits than your state requires.</p>

      <h2>Factors That Affect Your Premium</h2>
      <p>Insurance companies consider many factors when calculating your premium:</p>

      <ul>
        <li>Your age and driving experience</li>
        <li>Your driving record</li>
        <li>Your credit score (in most states)</li>
        <li>The type and value of your vehicle</li>
        <li>Where you live and park your car</li>
        <li>How much you drive</li>
        <li>Your coverage limits and deductibles</li>
      </ul>

      <h2>Money-Saving Tips</h2>
      <p>Here are proven strategies to reduce your auto insurance costs:</p>

      <ol>
        <li><strong>Shop around:</strong> Compare quotes from multiple insurers annually</li>
        <li><strong>Bundle policies:</strong> Combine auto and home insurance for discounts</li>
        <li><strong>Maintain good credit:</strong> A better credit score often means lower premiums</li>
        <li><strong>Take advantage of discounts:</strong> Ask about safe driver, student, military, and other discounts</li>
        <li><strong>Consider higher deductibles:</strong> Increasing your deductible can lower your premium</li>
        <li><strong>Drive safely:</strong> Maintain a clean driving record</li>
      </ol>

      <h2>The Claims Process</h2>
      <p>Understanding how to file a claim can save you time and stress when you need it most:</p>

      <h3>Steps to Take After an Accident</h3>
      <ol>
        <li>Ensure everyone's safety and call 911 if needed</li>
        <li>Exchange information with other drivers</li>
        <li>Document the scene with photos</li>
        <li>Get witness contact information</li>
        <li>File a police report if required</li>
        <li>Contact your insurance company as soon as possible</li>
      </ol>

      <h2>Conclusion</h2>
      <p>Auto insurance doesn't have to be confusing. By understanding the basics, knowing what coverage you need, and shopping smart, you can find the right policy at the right price. Remember, the cheapest option isn't always the best – focus on finding the right balance of coverage and cost for your specific situation.</p>

      <p>If you have questions about auto insurance or need help finding the right policy, our team at SecureAuto Hub is here to help. Contact us today for a free consultation and personalized quotes.</p>
    `
  }

  const relatedPosts = [
    {
      title: 'How Your Credit Score Affects Your Insurance Rates',
      slug: 'credit-score-insurance-rates',
      image: 'https://images.unsplash.com/photo-1554224155-6726b3ff858f?w=200&h=120&fit=crop'
    },
    {
      title: '10 Ways to Lower Your Auto Insurance Premiums',
      slug: 'lower-auto-insurance-premiums',
      image: 'https://images.unsplash.com/photo-1579621970563-ebec7560ff3e?w=200&h=120&fit=crop'
    },
    {
      title: 'Understanding Deductibles: How to Choose the Right Amount',
      slug: 'understanding-insurance-deductibles',
      image: 'https://images.unsplash.com/photo-1554224154-26032ffc0d07?w=200&h=120&fit=crop'
    }
  ]

  return (
    <div className="min-h-screen bg-gray-50">
      <article className="max-w-4xl mx-auto px-4 py-16">
        {/* Breadcrumb */}
        <nav className="mb-8">
          <ol className="flex text-sm text-gray-500">
            <li><Link href="/" className="hover:text-blue-600">Home</Link></li>
            <li className="mx-2">/</li>
            <li><Link href="/blog" className="hover:text-blue-600">Blog</Link></li>
            <li className="mx-2">/</li>
            <li className="text-gray-700">{post.title}</li>
          </ol>
        </nav>

        {/* Article Header */}
        <header className="mb-8">
          <div className="mb-4">
            <span className="bg-blue-100 text-blue-800 text-sm font-medium px-3 py-1 rounded-full">
              {post.category}
            </span>
          </div>
          <h1 className="text-4xl font-bold text-gray-900 mb-4">{post.title}</h1>
          <p className="text-xl text-gray-600 mb-6">{post.excerpt}</p>
          <div className="flex items-center text-gray-500 mb-6">
            <span>By {post.author}</span>
            <span className="mx-2">•</span>
            <span>{post.date}</span>
            <span className="mx-2">•</span>
            <span>{post.readTime}</span>
          </div>
          <Image
            src={post.image}
            alt={post.title}
            width={1200}
            height={600}
            className="w-full rounded-lg shadow-lg"
          />
        </header>

        {/* Article Content */}
        <div className="bg-white rounded-lg shadow-md p-8 mb-12">
          <div
            className="prose max-w-none text-gray-700 leading-relaxed"
            dangerouslySetInnerHTML={{ __html: post.content }}
            style={{
              lineHeight: '1.8',
            }}
          />
        </div>

        {/* Author Bio */}
        <div className="bg-white rounded-lg shadow-md p-6 mb-12">
          <div className="flex items-start space-x-4">
            <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center">
              <span className="text-blue-600 font-bold text-lg">
                {post.author.split(' ').map(n => n[0]).join('')}
              </span>
            </div>
            <div>
              <h3 className="text-lg font-semibold text-gray-900">{post.author}</h3>
              <p className="text-gray-600">
                Sarah is the founder and CEO of SecureAuto Hub, with over 15 years of experience in the insurance industry.
                She's passionate about helping people understand insurance and find the best coverage for their needs.
              </p>
            </div>
          </div>
        </div>

        {/* Related Posts */}
        <div className="bg-white rounded-lg shadow-md p-6">
          <h3 className="text-2xl font-bold text-gray-900 mb-6">Related Articles</h3>
          <div className="grid md:grid-cols-3 gap-6">
            {relatedPosts.map((relatedPost, index) => (
              <Link key={index} href={`/blog/${relatedPost.slug}`} className="group">
                <div className="bg-gray-50 rounded-lg overflow-hidden hover:shadow-md transition-shadow">
                  <Image
                    src={relatedPost.image}
                    alt={relatedPost.title}
                    width={200}
                    height={120}
                    className="w-full h-24 object-cover group-hover:scale-105 transition-transform duration-300"
                  />
                  <div className="p-4">
                    <h4 className="font-semibold text-gray-900 group-hover:text-blue-600 transition-colors">
                      {relatedPost.title}
                    </h4>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </div>

        {/* CTA Section */}
        <div className="mt-12 bg-blue-50 rounded-lg p-8 text-center">
          <h3 className="text-2xl font-bold text-gray-900 mb-4">
            Ready to Get Your Auto Insurance Quote?
          </h3>
          <p className="text-gray-600 mb-6">
            Let our experts help you find the perfect coverage at the best price.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Link
              href="/quote"
              className="bg-blue-600 text-white px-6 py-3 rounded-lg font-semibold hover:bg-blue-700 transition-colors"
            >
              Get Free Quote
            </Link>
            <Link
              href="/contact"
              className="bg-white text-blue-600 border border-blue-600 px-6 py-3 rounded-lg font-semibold hover:bg-blue-50 transition-colors"
            >
              Contact Expert
            </Link>
          </div>
        </div>
      </article>
    </div>
  )
}
